/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.apache.hc.client5.testing.extension.async;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.function.Consumer;

import org.apache.hc.client5.http.config.TlsConfig;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.client5.http.impl.cache.CachingHttpAsyncClientBuilder;
import org.apache.hc.client5.http.impl.nio.PoolingAsyncClientConnectionManagerBuilder;
import org.apache.hc.client5.http.ssl.DefaultClientTlsStrategy;
import org.apache.hc.core5.http2.HttpVersionPolicy;
import org.apache.hc.core5.io.CloseMode;
import org.apache.hc.core5.ssl.SSLContexts;
import org.junit.jupiter.api.extension.AfterEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;

public class CachingHttpAsyncClientResource implements AfterEachCallback {

    private final CachingHttpAsyncClientBuilder clientBuilder;
    private CloseableHttpAsyncClient client;

    public CachingHttpAsyncClientResource(final HttpVersionPolicy versionPolicy) throws IOException {
        this.clientBuilder = CachingHttpAsyncClientBuilder.create();
        try {
            this.clientBuilder
                    .setConnectionManager(PoolingAsyncClientConnectionManagerBuilder.create()
                            .setTlsStrategy(new DefaultClientTlsStrategy(SSLContexts.custom()
                                    .loadTrustMaterial(getClass().getResource("/test-ca.keystore"), "nopassword".toCharArray())
                                    .build()))
                            .setDefaultTlsConfig(TlsConfig.custom()
                                    .setVersionPolicy(versionPolicy)
                                    .build())
                            .build());
        } catch (final CertificateException | NoSuchAlgorithmException | KeyStoreException | KeyManagementException ex) {
            throw new IllegalStateException(ex);
        }
    }

    public void configure(final Consumer<CachingHttpAsyncClientBuilder> customizer) {
        customizer.accept(clientBuilder);
    }

    @Override
    public void afterEach(final ExtensionContext extensionContext) {
        if (client != null) {
            client.close(CloseMode.GRACEFUL);
        }
    }

    public CloseableHttpAsyncClient client() {
        if (client == null) {
            client = clientBuilder.build();
            client.start();
        }
        return client;
    }

}
